<?php
/**
 * JadCoins Product & Cart Points Display - V2.1
 * Shows estimated points on product page, cart, mini-cart, and all cart widgets
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class JadCoins_Product {

    private static $instance = null;
    private $loyalty_settings = null;
    private $product_points_displayed = false;

    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $api = JadCoins_API::instance();
        if ( ! $api->is_configured() ) {
            return;
        }

        // Product page points - multiple hooks for maximum theme compatibility
        if ( get_option( 'jadcoins_show_on_product', 1 ) ) {
            add_action( 'woocommerce_single_product_summary', array( $this, 'display_product_points' ), 11 );
            add_action( 'woocommerce_single_product_summary', array( $this, 'display_product_points' ), 25 );
            add_action( 'woocommerce_after_add_to_cart_form', array( $this, 'display_product_points' ), 10 );
            add_filter( 'woocommerce_get_price_html', array( $this, 'append_points_to_price' ), 100, 2 );
            add_shortcode( 'jadcoins_product_points', array( $this, 'shortcode_product_points' ) );
        }

        // Cart points - ALL possible cart locations
        if ( get_option( 'jadcoins_show_in_cart', 1 ) ) {
            add_action( 'woocommerce_cart_totals_before_order_total', array( $this, 'display_cart_points' ) );
            add_action( 'woocommerce_review_order_before_order_total', array( $this, 'display_cart_points' ) );
            add_action( 'woocommerce_mini_cart_contents', array( $this, 'display_mini_cart_points' ), 999 );
            add_action( 'woocommerce_after_mini_cart', array( $this, 'display_mini_cart_points_after' ) );
            add_action( 'woocommerce_widget_shopping_cart_before_buttons', array( $this, 'display_widget_cart_points' ) );
        }

        // Enqueue styles globally (mini cart can appear on any page)
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
    }

    public function enqueue_assets() {
        wp_enqueue_style(
            'jadcoins-product',
            JADCOINS_WOO_PLUGIN_URL . 'assets/css/product.css',
            array(),
            JADCOINS_WOO_VERSION
        );
    }

    private function get_loyalty() {
        if ( $this->loyalty_settings === null ) {
            $this->loyalty_settings = JadCoins_API::get_loyalty_settings();
        }
        return $this->loyalty_settings;
    }

    private function calculate_points( $amount ) {
        $loyalty = $this->get_loyalty();
        if ( empty( $loyalty ) ) {
            return 0;
        }
        $type = $loyalty['points_type'] ?? 'percent';
        if ( $type === 'percent' ) {
            $percentage = (float) ( $loyalty['percentage_points'] ?? 0 );
            return floor( $amount * ( $percentage / 100 ) );
        }
        return (int) ( $loyalty['fixed_points'] ?? 0 );
    }

    /**
     * Display points on single product page (prevents duplicates)
     */
    public function display_product_points() {
        if ( $this->product_points_displayed ) {
            return;
        }

        global $product;
        if ( ! $product || ! is_a( $product, 'WC_Product' ) ) {
            return;
        }

        $price = (float) wc_get_price_including_tax( $product );
        if ( $price <= 0 ) {
            return;
        }

        $points = $this->calculate_points( $price );
        if ( $points <= 0 ) {
            return;
        }

        $this->product_points_displayed = true;
        ?>
        <div class="jadcoins-product-points">
            <span class="jadcoins-product-points-icon"><?php echo JadCoins_API::get_icon_html( 20 ); ?></span>
            <span class="jadcoins-product-points-text">
                <?php
                printf(
                    esc_html__( 'Earn %s points with this purchase!', 'jadcoins-woo' ),
                    '<strong>' . esc_html( $points ) . '</strong>'
                );
                ?>
            </span>
        </div>
        <?php
    }

    /**
     * Append points inline after price HTML (single product only)
     */
    public function append_points_to_price( $price_html, $product ) {
        if ( ! is_product() || $this->product_points_displayed ) {
            return $price_html;
        }

        // Only main product, not related/upsells
        global $post;
        if ( ! $post || ! $product || $product->get_id() !== $post->ID ) {
            return $price_html;
        }

        $p = (float) wc_get_price_including_tax( $product );
        if ( $p <= 0 ) {
            return $price_html;
        }

        $points = $this->calculate_points( $p );
        if ( $points <= 0 ) {
            return $price_html;
        }

        $this->product_points_displayed = true;

        return $price_html . '<div class="jadcoins-price-points-inline">' . JadCoins_API::get_icon_html( 16 ) . ' '
            . sprintf( esc_html__( 'Earn %s points', 'jadcoins-woo' ), '<strong>' . $points . '</strong>' )
            . '</div>';
    }

    /**
     * Shortcode: [jadcoins_product_points]
     */
    public function shortcode_product_points( $atts ) {
        ob_start();
        $this->product_points_displayed = false; // Allow shortcode to force display
        $this->display_product_points();
        return ob_get_clean();
    }

    /**
     * Get cart points
     */
    private function get_cart_points() {
        if ( ! WC()->cart || WC()->cart->is_empty() ) {
            return 0;
        }
        $total = (float) WC()->cart->get_cart_contents_total() + (float) WC()->cart->get_cart_contents_tax();
        return $total > 0 ? $this->calculate_points( $total ) : 0;
    }

    /**
     * Cart page & checkout table row
     */
    public function display_cart_points() {
        $points = $this->get_cart_points();
        if ( $points <= 0 ) {
            return;
        }
        ?>
        <tr class="jadcoins-cart-points-row">
            <th><?php echo JadCoins_API::get_icon_html( 16 ); ?> <?php esc_html_e( 'JadCoins Points', 'jadcoins-woo' ); ?></th>
            <td><span class="jadcoins-cart-points-badge">+<?php echo esc_html( $points ); ?> <?php esc_html_e( 'points', 'jadcoins-woo' ); ?></span></td>
        </tr>
        <?php
    }

    /**
     * Mini cart contents (sidebar/dropdown)
     */
    public function display_mini_cart_points() {
        $points = $this->get_cart_points();
        if ( $points <= 0 ) {
            return;
        }
        ?>
        <li class="jadcoins-mini-cart-points">
            <span class="jadcoins-mini-cart-badge"><?php echo JadCoins_API::get_icon_html( 14 ); ?> +<?php echo esc_html( $points ); ?> <?php esc_html_e( 'points', 'jadcoins-woo' ); ?></span>
        </li>
        <?php
    }

    /**
     * After mini cart
     */
    public function display_mini_cart_points_after() {
        $points = $this->get_cart_points();
        if ( $points <= 0 ) {
            return;
        }
        ?>
        <div class="jadcoins-mini-cart-points-after">
            <?php echo JadCoins_API::get_icon_html( 16 ); ?> <?php printf( esc_html__( 'You will earn %s points with this order!', 'jadcoins-woo' ), '<strong>' . esc_html( $points ) . '</strong>' ); ?>
        </div>
        <?php
    }

    /**
     * Widget cart before buttons
     */
    public function display_widget_cart_points() {
        $points = $this->get_cart_points();
        if ( $points <= 0 ) {
            return;
        }
        ?>
        <div class="jadcoins-widget-cart-points">
            <?php echo JadCoins_API::get_icon_html( 16 ); ?> +<?php echo esc_html( $points ); ?> <?php esc_html_e( 'points', 'jadcoins-woo' ); ?>
        </div>
        <?php
    }
}
