<?php
/**
 * JadCoins Orders Integration - V2
 * Points calculated on: total after tax, excluding shipping
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class JadCoins_Orders {

    private static $instance = null;

    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        if ( ! get_option( 'jadcoins_enable_earn', 1 ) ) {
            return;
        }
        $status = get_option( 'jadcoins_earn_order_status', 'completed' );
        add_action( 'woocommerce_order_status_' . $status, array( $this, 'earn_points_on_order' ), 10, 1 );
        add_action( 'woocommerce_admin_order_data_after_billing_address', array( $this, 'display_order_meta' ), 10, 1 );
        add_action( 'add_meta_boxes', array( $this, 'add_order_meta_box' ) );
    }

    /**
     * Calculate purchase amount for points: total with tax, without shipping
     */
    private function get_purchase_amount( $order ) {
        // Items total + items tax (excludes shipping and shipping tax)
        $amount = (float) $order->get_subtotal();         // items before tax
        $amount += (float) $order->get_total_tax();        // all tax
        $amount -= (float) $order->get_shipping_tax();     // remove shipping tax

        // Also exclude any wallet discount that was applied
        $wallet_used = (float) $order->get_meta( '_jadcoins_wallet_amount' );
        // Don't subtract wallet - JadCoins API should calculate points on full amount before discount

        return max( $amount, 0 );
    }

    /**
     * Earn points when order reaches target status
     */
    public function earn_points_on_order( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) {
            return;
        }

        if ( $order->get_meta( '_jadcoins_points_processed' ) === 'yes' ) {
            return;
        }

        $api = JadCoins_API::instance();
        if ( ! $api->is_configured() ) {
            return;
        }

        // Get phone
        $user_id    = $order->get_customer_id();
        $phone      = '';
        $phone_code = '';

        if ( $user_id ) {
            $phone_data = JadCoins_API::get_user_phone( $user_id );
            $phone      = $phone_data['phone'];
            $phone_code = $phone_data['phone_code'];
        }

        if ( empty( $phone ) ) {
            $phone      = $order->get_billing_phone();
            $phone_code = $order->get_meta( '_billing_jadcoins_phone_code' );
            if ( empty( $phone_code ) ) {
                $phone_code = get_option( 'jadcoins_default_phone_code', '+972' );
            }
            $phone      = preg_replace( '/[^\d]/', '', $phone );
            $clean_code = preg_replace( '/[^\d]/', '', $phone_code );
            if ( ! empty( $clean_code ) && strpos( $phone, $clean_code ) === 0 ) {
                $phone = substr( $phone, strlen( $clean_code ) );
            }
        }

        if ( empty( $phone ) ) {
            $order->add_order_note( __( 'JadCoins: Could not earn points - no phone number found.', 'jadcoins-woo' ) );
            return;
        }

        // Ensure customer exists
        $customer_name  = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
        $customer_email = $order->get_billing_email();
        $birthday       = '';
        if ( $user_id ) {
            $birthday = JadCoins_API::get_user_birthday( $user_id );
        }
        $api->check_or_create_customer( $phone, $phone_code, $customer_name, $customer_email, $birthday );

        // Calculate: total after tax, minus shipping
        $purchase_amount = $this->get_purchase_amount( $order );

        $result = $api->earn_points( $phone, $phone_code, $purchase_amount, 'WC-' . $order_id );

        if ( ! empty( $result['value'] ) && ! empty( $result['data'] ) ) {
            $data = $result['data'];
            $order->update_meta_data( '_jadcoins_points_processed', 'yes' );
            $order->update_meta_data( '_jadcoins_points_earned', $data['points_earned'] ?? 0 );
            $order->update_meta_data( '_jadcoins_points_new_balance', $data['new_balance'] ?? 0 );
            $order->update_meta_data( '_jadcoins_earn_transaction_id', $data['transaction_id'] ?? '' );
            $order->update_meta_data( '_jadcoins_purchase_amount_sent', $purchase_amount );

            if ( ! empty( $data['auto_redeemed'] ) ) {
                $order->update_meta_data( '_jadcoins_auto_redeemed', wp_json_encode( $data['auto_redeemed'] ) );
            }
            $order->save();

            $order->add_order_note( sprintf(
                __( 'JadCoins: %1$s points earned (on %2$s). New balance: %3$s. TXN: %4$s', 'jadcoins-woo' ),
                $data['points_earned'] ?? 0,
                wc_price( $purchase_amount ),
                $data['new_balance'] ?? 0,
                $data['transaction_id'] ?? 'N/A'
            ) );

            if ( ! empty( $data['auto_redeemed'] ) ) {
                foreach ( $data['auto_redeemed'] as $r ) {
                    $order->add_order_note( sprintf(
                        __( 'JadCoins: Auto-redeemed %1$s points → %2$s added to wallet.', 'jadcoins-woo' ),
                        $r['redeemed_points'] ?? 0,
                        $r['wallet_value'] ?? 0
                    ) );
                }
            }
        } else {
            $order->add_order_note( sprintf(
                __( 'JadCoins: Failed to earn points - %s', 'jadcoins-woo' ),
                $result['error'] ?? 'Unknown error'
            ) );
        }
    }

    public function display_order_meta( $order ) {
        $points_earned = $order->get_meta( '_jadcoins_points_earned' );
        $wallet_used   = $order->get_meta( '_jadcoins_wallet_amount' );
        if ( $points_earned || $wallet_used ) {
            echo '<div style="margin-top:15px;padding:10px;background:#fffbeb;border:1px solid #f59e0b;border-radius:4px;">';
            echo '<h4 style="margin:0 0 8px;color:#b45309;">' . JadCoins_API::get_icon_html( 18 ) . ' JadCoins</h4>';
            if ( $wallet_used ) {
                printf( '<p style="margin:2px 0;"><strong>%s</strong> %s</p>', esc_html__( 'Wallet Used:', 'jadcoins-woo' ), wp_kses_post( wc_price( $wallet_used ) ) );
            }
            if ( $points_earned ) {
                printf( '<p style="margin:2px 0;"><strong>%s</strong> %s</p>', esc_html__( 'Points Earned:', 'jadcoins-woo' ), esc_html( $points_earned ) );
            }
            echo '</div>';
        }
    }

    public function add_order_meta_box() {
        $screen = 'shop_order';
        if ( class_exists( '\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController' ) ) {
            try {
                $controller = wc_get_container()->get( \Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class );
                if ( $controller->custom_orders_table_usage_is_enabled() ) {
                    $screen = wc_get_page_screen_id( 'shop-order' );
                }
            } catch ( \Exception $e ) {
                // Fallback
            }
        }
        add_meta_box( 'jadcoins-order-info', __( 'JadCoins Loyalty', 'jadcoins-woo' ), array( $this, 'render_order_meta_box' ), $screen, 'side', 'default' );
    }

    public function render_order_meta_box( $post_or_order ) {
        $order = ( $post_or_order instanceof WP_Post ) ? wc_get_order( $post_or_order->ID ) : $post_or_order;
        if ( ! $order ) {
            return;
        }

        $points_processed = $order->get_meta( '_jadcoins_points_processed' );
        $points_earned    = $order->get_meta( '_jadcoins_points_earned' );
        $wallet_used      = $order->get_meta( '_jadcoins_wallet_amount' );
        $wallet_processed = $order->get_meta( '_jadcoins_wallet_processed' );
        $earn_txn         = $order->get_meta( '_jadcoins_earn_transaction_id' );
        $wallet_txn       = $order->get_meta( '_jadcoins_wallet_transaction_id' );
        $amount_sent      = $order->get_meta( '_jadcoins_purchase_amount_sent' );

        echo '<div style="line-height:1.8;">';
        if ( $wallet_used ) {
            $icon = $wallet_processed === 'yes' ? '✅' : '⏳';
            printf( '<p>%s <strong>%s</strong> %s</p>', $icon, esc_html__( 'Wallet Used:', 'jadcoins-woo' ), wp_kses_post( wc_price( $wallet_used ) ) );
            if ( $wallet_txn ) {
                printf( '<p style="color:#666;font-size:12px;">TXN: %s</p>', esc_html( $wallet_txn ) );
            }
        }
        if ( $points_processed === 'yes' ) {
            printf( '<p>✅ <strong>%s</strong> %s</p>', esc_html__( 'Points Earned:', 'jadcoins-woo' ), esc_html( $points_earned ) );
            if ( $amount_sent ) {
                printf( '<p style="color:#666;font-size:12px;">%s: %s</p>', esc_html__( 'Calculated on', 'jadcoins-woo' ), wp_kses_post( wc_price( $amount_sent ) ) );
            }
            if ( $earn_txn ) {
                printf( '<p style="color:#666;font-size:12px;">TXN: %s</p>', esc_html( $earn_txn ) );
            }
        } else {
            printf( '<p style="color:#999;">%s</p>', esc_html__( 'Points not yet earned (order not completed)', 'jadcoins-woo' ) );
        }
        if ( ! $wallet_used && $points_processed !== 'yes' ) {
            printf( '<p style="color:#999;">%s</p>', esc_html__( 'No JadCoins activity for this order.', 'jadcoins-woo' ) );
        }
        echo '</div>';
    }
}
