<?php
/**
 * JadCoins Checkout Integration
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class JadCoins_Checkout {

    private static $instance = null;

    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        // Only if wallet is enabled
        if ( ! get_option( 'jadcoins_enable_wallet', 1 ) ) {
            return;
        }

        // Enqueue frontend assets
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );

        // Display wallet option in checkout
        add_action( 'woocommerce_review_order_before_payment', array( $this, 'display_wallet_option' ) );

        // AJAX handlers
        add_action( 'wp_ajax_jadcoins_apply_wallet', array( $this, 'ajax_apply_wallet' ) );
        add_action( 'wp_ajax_jadcoins_remove_wallet', array( $this, 'ajax_remove_wallet' ) );
        add_action( 'wp_ajax_jadcoins_get_balance', array( $this, 'ajax_get_balance' ) );

        // Apply wallet discount to cart
        add_action( 'woocommerce_cart_calculate_fees', array( $this, 'apply_wallet_fee' ) );

        // Store wallet info in order meta
        add_action( 'woocommerce_checkout_create_order', array( $this, 'save_wallet_to_order' ), 10, 2 );

        // Process wallet deduction after order placed
        add_action( 'woocommerce_checkout_order_processed', array( $this, 'process_wallet_deduction' ), 10, 3 );

        // Clear session on thank you page
        add_action( 'woocommerce_thankyou', array( $this, 'clear_wallet_session' ) );
    }

    /**
     * Enqueue checkout assets
     */
    public function enqueue_assets() {
        if ( ! is_checkout() ) {
            return;
        }

        wp_enqueue_style(
            'jadcoins-checkout',
            JADCOINS_WOO_PLUGIN_URL . 'assets/css/checkout.css',
            array(),
            JADCOINS_WOO_VERSION
        );

        wp_enqueue_script(
            'jadcoins-checkout',
            JADCOINS_WOO_PLUGIN_URL . 'assets/js/checkout.js',
            array( 'jquery', 'wc-checkout' ),
            JADCOINS_WOO_VERSION,
            true
        );

        wp_localize_script( 'jadcoins-checkout', 'jadcoins_checkout', array(
            'ajax_url'     => admin_url( 'admin-ajax.php' ),
            'nonce'        => wp_create_nonce( 'jadcoins_checkout_nonce' ),
            'currency'     => get_woocommerce_currency_symbol(),
            'i18n'         => array(
                'loading'       => __( 'Loading...', 'jadcoins-woo' ),
                'apply'         => __( 'Apply', 'jadcoins-woo' ),
                'remove'        => __( 'Remove', 'jadcoins-woo' ),
                'applied'       => __( 'Wallet discount applied!', 'jadcoins-woo' ),
                'removed'       => __( 'Wallet discount removed.', 'jadcoins-woo' ),
                'error'         => __( 'Something went wrong. Please try again.', 'jadcoins-woo' ),
                'no_balance'    => __( 'No wallet balance available.', 'jadcoins-woo' ),
                'enter_amount'  => __( 'Enter amount to use', 'jadcoins-woo' ),
                'max_exceeded'  => __( 'Amount exceeds your wallet balance.', 'jadcoins-woo' ),
            ),
        ) );
    }

    /**
     * Display wallet option at checkout
     */
    public function display_wallet_option() {
        if ( ! is_user_logged_in() ) {
            return;
        }

        $api = JadCoins_API::instance();
        if ( ! $api->is_configured() ) {
            return;
        }

        $wallet_applied = WC()->session ? WC()->session->get( 'jadcoins_wallet_amount', 0 ) : 0;
        ?>
        <div id="jadcoins-wallet-section" class="jadcoins-wallet-box">
            <h3 class="jadcoins-wallet-title">
                <span class="jadcoins-icon"><?php echo JadCoins_API::get_icon_html( 22 ); ?></span>
                <?php esc_html_e( 'JadCoins Wallet', 'jadcoins-woo' ); ?>
            </h3>

            <div id="jadcoins-wallet-content">
                <div id="jadcoins-wallet-loading" style="display:none;">
                    <span class="jadcoins-spinner"></span>
                    <?php esc_html_e( 'Loading your balance...', 'jadcoins-woo' ); ?>
                </div>

                <div id="jadcoins-wallet-balance-info" style="display:none;">
                    <div class="jadcoins-balance-row">
                        <span class="jadcoins-balance-label"><?php esc_html_e( 'Points Balance:', 'jadcoins-woo' ); ?></span>
                        <span id="jadcoins-points-balance" class="jadcoins-balance-value">0</span>
                    </div>
                    <div class="jadcoins-balance-row">
                        <span class="jadcoins-balance-label"><?php esc_html_e( 'Wallet Balance:', 'jadcoins-woo' ); ?></span>
                        <span id="jadcoins-wallet-balance" class="jadcoins-balance-value">0</span>
                    </div>
                </div>

                <div id="jadcoins-wallet-form" style="display:none;">
                    <div class="jadcoins-input-group">
                        <input type="number" id="jadcoins-wallet-amount"
                               min="0" step="0.01"
                               placeholder="<?php esc_attr_e( 'Amount to use', 'jadcoins-woo' ); ?>" />
                        <button type="button" id="jadcoins-apply-wallet" class="button jadcoins-btn-apply">
                            <?php esc_html_e( 'Apply', 'jadcoins-woo' ); ?>
                        </button>
                    </div>
                    <p id="jadcoins-wallet-message" class="jadcoins-message" style="display:none;"></p>
                </div>

                <div id="jadcoins-wallet-applied" style="<?php echo $wallet_applied > 0 ? '' : 'display:none;'; ?>">
                    <div class="jadcoins-applied-row">
                        <span class="jadcoins-applied-text">
                            <?php esc_html_e( 'Wallet discount:', 'jadcoins-woo' ); ?>
                            <strong id="jadcoins-applied-amount"><?php echo $wallet_applied > 0 ? wc_price( $wallet_applied ) : ''; ?></strong>
                        </span>
                        <button type="button" id="jadcoins-remove-wallet" class="jadcoins-btn-remove">
                            <?php esc_html_e( 'Remove', 'jadcoins-woo' ); ?>
                        </button>
                    </div>
                </div>

                <div id="jadcoins-wallet-unavailable" style="display:none;">
                    <p class="jadcoins-unavailable-text"><?php esc_html_e( 'No wallet balance available.', 'jadcoins-woo' ); ?></p>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * AJAX: Get customer balance
     */
    public function ajax_get_balance() {
        check_ajax_referer( 'jadcoins_checkout_nonce', 'nonce' );

        if ( ! is_user_logged_in() ) {
            wp_send_json_error( 'Not logged in' );
        }

        $api       = JadCoins_API::instance();
        $user_id   = get_current_user_id();
        $phone_data = JadCoins_API::get_user_phone( $user_id );

        if ( empty( $phone_data['phone'] ) ) {
            wp_send_json_error( __( 'Phone number not found. Please update your billing details.', 'jadcoins-woo' ) );
        }

        // Try to check-or-create customer first
        $user = get_userdata( $user_id );
        $result = $api->check_or_create_customer(
            $phone_data['phone'],
            $phone_data['phone_code'],
            $user->display_name,
            $user->user_email
        );

        if ( ! empty( $result['value'] ) && ! empty( $result['data'] ) ) {
            $data = $result['data'];
            wp_send_json_success( array(
                'points_balance' => $data['wallets']['points_balance'] ?? 0,
                'wallet_balance' => $data['wallets']['wallet_balance'] ?? 0,
                'points_value'   => $data['wallets']['points_value'] ?? 0,
                'currency'       => $data['currency'] ?? '₪',
            ) );
        } else {
            wp_send_json_error( $result['error'] ?? __( 'Could not retrieve balance.', 'jadcoins-woo' ) );
        }
    }

    /**
     * AJAX: Apply wallet amount
     */
    public function ajax_apply_wallet() {
        check_ajax_referer( 'jadcoins_checkout_nonce', 'nonce' );

        if ( ! is_user_logged_in() ) {
            wp_send_json_error( 'Not logged in' );
        }

        $amount = isset( $_POST['amount'] ) ? (float) sanitize_text_field( $_POST['amount'] ) : 0;

        if ( $amount <= 0 ) {
            wp_send_json_error( __( 'Please enter a valid amount.', 'jadcoins-woo' ) );
        }

        // Validate against cart total
        $cart_total = WC()->cart ? WC()->cart->get_subtotal() : 0;
        if ( $amount > $cart_total ) {
            $amount = $cart_total;
        }

        // Validate against wallet balance
        $api        = JadCoins_API::instance();
        $phone_data = JadCoins_API::get_user_phone();

        if ( empty( $phone_data['phone'] ) ) {
            wp_send_json_error( __( 'Phone number not found.', 'jadcoins-woo' ) );
        }

        $balance_result = $api->get_customer_balance( $phone_data['phone'], $phone_data['phone_code'] );

        if ( empty( $balance_result['value'] ) ) {
            wp_send_json_error( __( 'Could not verify balance.', 'jadcoins-woo' ) );
        }

        $wallet_balance = (float) ( $balance_result['data']['balances']['wallet_balance'] ?? 0 );

        if ( $amount > $wallet_balance ) {
            wp_send_json_error(
                sprintf(
                    /* translators: %s: wallet balance amount */
                    __( 'Amount exceeds wallet balance. Maximum: %s', 'jadcoins-woo' ),
                    wc_price( $wallet_balance )
                )
            );
        }

        // Store in session
        WC()->session->set( 'jadcoins_wallet_amount', $amount );
        WC()->session->set( 'jadcoins_phone', $phone_data['phone'] );
        WC()->session->set( 'jadcoins_phone_code', $phone_data['phone_code'] );

        wp_send_json_success( array(
            'amount'    => $amount,
            'formatted' => wc_price( $amount ),
        ) );
    }

    /**
     * AJAX: Remove wallet discount
     */
    public function ajax_remove_wallet() {
        check_ajax_referer( 'jadcoins_checkout_nonce', 'nonce' );

        if ( WC()->session ) {
            WC()->session->set( 'jadcoins_wallet_amount', 0 );
            WC()->session->__unset( 'jadcoins_phone' );
            WC()->session->__unset( 'jadcoins_phone_code' );
        }

        wp_send_json_success();
    }

    /**
     * Apply wallet as negative fee in cart
     */
    public function apply_wallet_fee( $cart ) {
        if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
            return;
        }

        if ( ! WC()->session ) {
            return;
        }

        $amount = (float) WC()->session->get( 'jadcoins_wallet_amount', 0 );

        if ( $amount > 0 ) {
            $cart->add_fee(
                __( 'JadCoins Wallet Discount', 'jadcoins-woo' ),
                -$amount,
                false
            );
        }
    }

    /**
     * Save wallet info to order meta
     */
    public function save_wallet_to_order( $order, $data ) {
        if ( ! WC()->session ) {
            return;
        }

        $amount = (float) WC()->session->get( 'jadcoins_wallet_amount', 0 );

        if ( $amount > 0 ) {
            $order->update_meta_data( '_jadcoins_wallet_amount', $amount );
            $order->update_meta_data( '_jadcoins_wallet_phone', WC()->session->get( 'jadcoins_phone', '' ) );
            $order->update_meta_data( '_jadcoins_wallet_phone_code', WC()->session->get( 'jadcoins_phone_code', '' ) );
            $order->update_meta_data( '_jadcoins_wallet_processed', 'no' );
        }
    }

    /**
     * Process wallet deduction via API after order placed
     */
    public function process_wallet_deduction( $order_id, $posted_data, $order ) {
        $amount     = (float) $order->get_meta( '_jadcoins_wallet_amount' );
        $phone      = $order->get_meta( '_jadcoins_wallet_phone' );
        $phone_code = $order->get_meta( '_jadcoins_wallet_phone_code' );
        $processed  = $order->get_meta( '_jadcoins_wallet_processed' );

        if ( $amount <= 0 || empty( $phone ) || $processed === 'yes' ) {
            return;
        }

        $api    = JadCoins_API::instance();
        $result = $api->use_wallet(
            $phone,
            $phone_code,
            $amount,
             (float) $order->get_total(), // ✅ amount paid after wallet discount
            'WC-ORDER-' . $order_id
        );

        if ( ! empty( $result['value'] ) && ! empty( $result['data'] ) ) {
            $order->update_meta_data( '_jadcoins_wallet_processed', 'yes' );
            $order->update_meta_data( '_jadcoins_wallet_transaction_id', $result['data']['wallet_transaction_id'] ?? '' );
            $order->update_meta_data( '_jadcoins_wallet_new_balance', $result['data']['new_wallet_balance'] ?? 0 );
            
            // ✅ إذا استخدم محفظة: خذ earned_points من use-wallet
            $order->update_meta_data( '_jadcoins_points_processed', 'yes' );
            $order->update_meta_data( '_jadcoins_points_earned', $result['data']['earned_points'] ?? 0 );
            
            // اختياري فقط إذا الـ API بيرجع رصيد نقاط جديد (حسب الريسبونس عندك)
            if ( isset( $result['data']['new_points_balance'] ) ) {
                $order->update_meta_data( '_jadcoins_points_new_balance', $result['data']['new_points_balance'] );
            }

            
            $order->save();

            $order->add_order_note(
                sprintf(
                    /* translators: 1: wallet amount, 2: transaction ID */
                   __( 'JadCoins wallet: %1$s deducted. Earned points: %2$s. Transaction ID: %3$s', 'jadcoins-woo' ),
                    wc_price( $amount ),
                    $result['data']['wallet_transaction_id'] ?? 'N/A'
                )
            );
        } else {
            $order->add_order_note(
                sprintf(
                    /* translators: %s: error message */
                    __( 'JadCoins wallet deduction failed: %s', 'jadcoins-woo' ),
                    $result['error'] ?? 'Unknown error'
                )
            );
        }
    }

    /**
     * Clear session data
     */
    public function clear_wallet_session( $order_id ) {
        if ( WC()->session ) {
            WC()->session->set( 'jadcoins_wallet_amount', 0 );
            WC()->session->__unset( 'jadcoins_phone' );
            WC()->session->__unset( 'jadcoins_phone_code' );
        }
    }
}
