<?php
/**
 * JadCoins API Handler - V2
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class JadCoins_API {

    private static $instance = null;
    private $api_url;
    private $company_code;

    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->api_url      = rtrim( get_option( 'jadcoins_api_url', 'https://jadcoins.online/api' ), '/' );
        $this->company_code = get_option( 'jadcoins_company_code', '' );
    }

    public function reload() {
        $this->api_url      = rtrim( get_option( 'jadcoins_api_url', 'https://jadcoins.online/api' ), '/' );
        $this->company_code = get_option( 'jadcoins_company_code', '' );
    }

    public function get_company_code() {
        return $this->company_code;
    }

    public function get_api_url() {
        return $this->api_url;
    }

    private function get( $endpoint, $params = array() ) {
        $url = $this->api_url . $endpoint;
        if ( ! empty( $params ) ) {
            $url = add_query_arg( $params, $url );
        }
        $response = wp_remote_get( $url, array(
            'timeout' => 30,
            'headers' => array( 'Accept' => 'application/json', 'Content-Type' => 'application/json' ),
        ) );
        return $this->handle_response( $response );
    }

    private function post( $endpoint, $body = array() ) {
        $url = $this->api_url . $endpoint;
        $response = wp_remote_post( $url, array(
            'timeout' => 30,
            'headers' => array( 'Accept' => 'application/json', 'Content-Type' => 'application/json' ),
            'body'    => wp_json_encode( $body ),
        ) );
        return $this->handle_response( $response );
    }

    private function handle_response( $response ) {
        if ( is_wp_error( $response ) ) {
            return array( 'value' => false, 'error' => $response->get_error_message() );
        }
        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'JadCoins API [' . $code . ']: ' . $body );
        }

        if ( $code >= 400 || empty( $data ) ) {
            $error_msg = $data['message'] ?? $data['error'] ?? '';
            if ( isset( $data['errors'] ) ) {
                $error_msg = is_array( $data['errors'] ) ? wp_json_encode( $data['errors'] ) : $data['errors'];
            }
            if ( empty( $error_msg ) ) {
                $error_msg = 'HTTP ' . $code;
            }
            return array( 'value' => false, 'error' => $error_msg );
        }
        return $data;
    }

    /**
     * GET /integration/company/check?unique_code=XXX
     */
    public function check_company( $code = '' ) {
        $code = $code ?: $this->company_code;
        return $this->get( '/integration/company/check', array( 'unique_code' => $code ) );
    }

    /**
     * POST /integration/customer/check-or-create
     * Now includes birthday field
     */
    public function check_or_create_customer( $phone, $phone_code, $name = '', $email = '', $birthday = '' ) {
        $body = array(
            'company_unique_code' => $this->company_code,
            'phone'              => $phone,
            'phone_code'         => $phone_code,
            'name'               => $name,
            'email'              => $email,
        );

        if ( ! empty( $birthday ) ) {
            $body['birth_date'] = $birthday; // Format: YYYY-MM-DD
        }

        return $this->post( '/integration/customer/check-or-create', $body );
    }

    /**
     * GET /integration/customer/balance
     */
    public function get_customer_balance( $phone, $phone_code ) {
        return $this->get( '/integration/customer/balance', array(
            'company_unique_code' => $this->company_code,
            'phone'              => $phone,
            'phone_code'         => $phone_code,
        ) );
    }

    /**
     * POST /integration/points/earn
     */
    public function earn_points( $phone, $phone_code, $purchase_amount, $order_id ) {
        return $this->post( '/integration/points/earn', array(
            'company_unique_code' => $this->company_code,
            'phone'              => $phone,
            'phone_code'         => $phone_code,
            'purchase_amount'    => (float) $purchase_amount,
            'order_id'           => (string) $order_id,
        ) );
    }

    /**
     * POST /integration/use-wallet
     */
    public function use_wallet( $phone, $phone_code, $amount, $purchase_amount, $reference ) {
        return $this->post( '/integration/use-wallet', array(
            'company_unique_code' => $this->company_code,
            'phone'              => $phone,
            'phone_code'         => $phone_code,
            'amount'             => (float) $amount,
            'purchase_amount'    => (float) $purchase_amount,
            'reference'          => (string) $reference,
        ) );
    }

    /**
     * Get phone data from WP user
     */
    public static function get_user_phone( $user_id = 0 ) {
        if ( ! $user_id ) {
            $user_id = get_current_user_id();
        }
        $phone      = get_user_meta( $user_id, 'billing_phone', true );
        $phone_code = get_user_meta( $user_id, 'jadcoins_phone_code', true );
        if ( empty( $phone_code ) ) {
            $phone_code = get_option( 'jadcoins_default_phone_code', '+972' );
        }
        if ( ! empty( $phone ) ) {
            $phone = preg_replace( '/[^\d]/', '', $phone );
            $clean_code = preg_replace( '/[^\d]/', '', $phone_code );
            if ( ! empty( $clean_code ) && strpos( $phone, $clean_code ) === 0 ) {
                $phone = substr( $phone, strlen( $clean_code ) );
            }
        }
        return array( 'phone' => $phone, 'phone_code' => $phone_code );
    }

    /**
     * Get user birthday
     */
    public static function get_user_birthday( $user_id = 0 ) {
        if ( ! $user_id ) {
            $user_id = get_current_user_id();
        }
        return get_user_meta( $user_id, 'jadcoins_birthday', true );
    }

    public function is_configured() {
        return ! empty( $this->api_url ) && ! empty( $this->company_code );
    }

    /**
     * Get JadCoins icon HTML for frontend
     */
    public static function get_icon_html( $size = 18 ) {
        return '<img src="' . esc_url( JADCOINS_WOO_PLUGIN_URL . 'assets/img/jadcoins-icon.png' ) . '" alt="JadCoins" class="jadcoins-logo-icon" style="width:' . intval( $size ) . 'px;height:' . intval( $size ) . 'px;vertical-align:middle;" />';
    }

    /**
     * Check if connection is established and locked
     */
    public static function is_connected() {
        // Support V1 users: if company data is cached, consider connected
        if ( get_option( 'jadcoins_connected', false ) ) {
            return true;
        }
        // Also check if we have cached company data (from a successful test)
        $data = get_option( 'jadcoins_company_data', array() );
        return ! empty( $data ) && ! empty( get_option( 'jadcoins_company_code', '' ) );
    }

    /**
     * Get cached company info
     */
    public static function get_cached_company() {
        return get_option( 'jadcoins_company_data', array() );
    }

    /**
     * Get loyalty settings from cache, or fetch live
     */
    public static function get_loyalty_settings() {
        $data = self::get_cached_company();
        if ( ! empty( $data['loyalty_settings'] ) ) {
            return $data['loyalty_settings'];
        }

        // Fallback: fetch from API and cache
        $api = self::instance();
        if ( ! $api->is_configured() ) {
            return array();
        }

        $result = $api->check_company();
        if ( ! empty( $result['value'] ) && ! empty( $result['data'] ) ) {
            update_option( 'jadcoins_company_data', $result['data'] );
            return $result['data']['loyalty_settings'] ?? array();
        }

        return array();
    }
}
