<?php
/**
 * JadCoins Admin Settings - V2
 * Features: Connection lock/disconnect, secure settings
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class JadCoins_Admin {

    private static $instance = null;

    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
        add_action( 'wp_ajax_jadcoins_test_connection', array( $this, 'ajax_test_connection' ) );
        add_action( 'wp_ajax_jadcoins_disconnect', array( $this, 'ajax_disconnect' ) );
    }

    public function add_admin_menu() {

        $icon_url = plugin_dir_url( dirname(__FILE__) ) . 'assets/img/jadcoins-menu-icon.png';

        add_menu_page(
            __( 'JadCoins', 'jadcoins-woo' ),
            __( 'JadCoins', 'jadcoins-woo' ),
            'manage_woocommerce',
            'jadcoins-settings',
            array( $this, 'settings_page' ),
            $icon_url,
            56
        );
    }



    public function register_settings() {
        register_setting( 'jadcoins_settings', 'jadcoins_api_url', array(
            'sanitize_callback' => 'esc_url_raw',
            'default'           => 'https://jadcoins.online/api',
        ) );
        register_setting( 'jadcoins_settings', 'jadcoins_company_code', 'sanitize_text_field' );
        register_setting( 'jadcoins_settings', 'jadcoins_default_phone_code', array(
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => '+972',
        ) );
        register_setting( 'jadcoins_settings', 'jadcoins_enable_earn', array(
            'sanitize_callback' => 'absint',
            'default'           => 1,
        ) );
        register_setting( 'jadcoins_settings', 'jadcoins_enable_wallet', array(
            'sanitize_callback' => 'absint',
            'default'           => 1,
        ) );
        register_setting( 'jadcoins_settings', 'jadcoins_earn_order_status', array(
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => 'completed',
        ) );
        register_setting( 'jadcoins_settings', 'jadcoins_show_in_myaccount', array(
            'sanitize_callback' => 'absint',
            'default'           => 1,
        ) );
        register_setting( 'jadcoins_settings', 'jadcoins_show_on_product', array(
            'sanitize_callback' => 'absint',
            'default'           => 1,
        ) );
        register_setting( 'jadcoins_settings', 'jadcoins_show_in_cart', array(
            'sanitize_callback' => 'absint',
            'default'           => 1,
        ) );
    }

    public function enqueue_admin_assets( $hook ) {
        if ( $hook !== 'toplevel_page_jadcoins-settings' ) {
            return;
        }
        wp_enqueue_style( 'jadcoins-admin', JADCOINS_WOO_PLUGIN_URL . 'assets/css/admin.css', array(), JADCOINS_WOO_VERSION );
        wp_enqueue_script( 'jadcoins-admin', JADCOINS_WOO_PLUGIN_URL . 'assets/js/admin.js', array( 'jquery' ), JADCOINS_WOO_VERSION, true );
        wp_localize_script( 'jadcoins-admin', 'jadcoins_admin', array(
            'ajax_url'     => admin_url( 'admin-ajax.php' ),
            'nonce'        => wp_create_nonce( 'jadcoins_admin_nonce' ),
            'is_connected' => JadCoins_API::is_connected(),
            'i18n'         => array(
                'testing'           => __( 'Testing connection...', 'jadcoins-woo' ),
                'success'           => __( 'Connection successful!', 'jadcoins-woo' ),
                'error'             => __( 'Connection failed:', 'jadcoins-woo' ),
                'confirm_disconnect'=> __( 'Are you sure you want to disconnect from JadCoins? This will remove the connection and you will need to reconnect.', 'jadcoins-woo' ),
                'disconnected'      => __( 'Disconnected successfully. You can now change settings.', 'jadcoins-woo' ),
            ),
        ) );
    }

    /**
     * AJAX: Test & lock connection
     */
    public function ajax_test_connection() {
        check_ajax_referer( 'jadcoins_admin_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $api_url      = isset( $_POST['api_url'] ) ? esc_url_raw( wp_unslash( $_POST['api_url'] ) ) : '';
        $company_code = isset( $_POST['company_code'] ) ? sanitize_text_field( wp_unslash( $_POST['company_code'] ) ) : '';

        if ( empty( $api_url ) || empty( $company_code ) ) {
            wp_send_json_error( __( 'Please fill in API URL and Company Code.', 'jadcoins-woo' ) );
        }

        $api_url  = rtrim( $api_url, '/' );
        $full_url = $api_url . '/integration/company/check?' . http_build_query( array( 'unique_code' => $company_code ) );

        $response = wp_remote_get( $full_url, array(
            'timeout' => 30,
            'headers' => array( 'Accept' => 'application/json', 'Content-Type' => 'application/json' ),
        ) );

        if ( is_wp_error( $response ) ) {
            wp_send_json_error( $response->get_error_message() );
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( $code >= 400 || empty( $data ) || empty( $data['value'] ) ) {
            $error_msg = $data['message'] ?? $data['error'] ?? '';
            if ( isset( $data['errors'] ) ) {
                $error_msg = is_array( $data['errors'] ) ? wp_json_encode( $data['errors'] ) : $data['errors'];
            }
            wp_send_json_error( ( $error_msg ?: 'HTTP ' . $code ) . ' | URL: ' . $full_url );
        }

        // Save and LOCK the connection
        update_option( 'jadcoins_api_url', $api_url );
        update_option( 'jadcoins_company_code', $company_code );
        update_option( 'jadcoins_connected', true );
        update_option( 'jadcoins_company_data', $data['data'] ?? array() );
        update_option( 'jadcoins_connected_at', current_time( 'mysql' ) );

        wp_send_json_success( array(
            'company_name'    => $data['data']['name'] ?? '',
            'loyalty_enabled' => $data['data']['loyalty_enabled'] ?? false,
            'settings'        => $data['data']['loyalty_settings'] ?? array(),
            'currency'        => $data['data']['currency'] ?? array(),
            'avatar'          => $data['data']['avatar'] ?? '',
        ) );
    }

    /**
     * AJAX: Disconnect
     */
    public function ajax_disconnect() {
        check_ajax_referer( 'jadcoins_admin_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        delete_option( 'jadcoins_connected' );
        delete_option( 'jadcoins_company_data' );
        delete_option( 'jadcoins_connected_at' );
        // Keep api_url and company_code so user can reconnect easily

        wp_send_json_success();
    }

    /**
     * Settings page
     */
    public function settings_page() {
        $is_connected  = JadCoins_API::is_connected();
        $company_data  = JadCoins_API::get_cached_company();
        $connected_at  = get_option( 'jadcoins_connected_at', '' );
        ?>
        <div class="wrap jadcoins-settings-wrap">
            <h1>
                <img src="<?php echo esc_url( JADCOINS_WOO_PLUGIN_URL . 'assets/img/jadcoins-icon.png' ); ?>" alt="JadCoins" style="width:30px;height:30px;vertical-align:middle;margin-right:8px;" />
                <?php esc_html_e( 'JadCoins Settings', 'jadcoins-woo' ); ?>
                <span class="jadcoins-version">v<?php echo esc_html( JADCOINS_WOO_VERSION ); ?></span>
            </h1>

            <form method="post" action="options.php" id="jadcoins-settings-form">
                <?php settings_fields( 'jadcoins_settings' ); ?>

                <!-- Connection Card -->
                <div class="jadcoins-card">
                    <h2><?php esc_html_e( 'API Connection', 'jadcoins-woo' ); ?></h2>

                    <?php if ( $is_connected && ! empty( $company_data ) ) : ?>
                        <!-- Connected State -->
                        <div class="jadcoins-connected-box">
                            <div class="jadcoins-connected-header">
                                <span class="jadcoins-connected-badge">✅ <?php esc_html_e( 'Connected', 'jadcoins-woo' ); ?></span>
                                <?php if ( $connected_at ) : ?>
                                    <span class="jadcoins-connected-date">
                                        <?php printf( esc_html__( 'Since %s', 'jadcoins-woo' ), esc_html( $connected_at ) ); ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                            <div class="jadcoins-connected-info">
                                <?php if ( ! empty( $company_data['avatar'] ) ) : ?>
                                    <img src="<?php echo esc_url( $company_data['avatar'] ); ?>" alt="" class="jadcoins-company-avatar" />
                                <?php endif; ?>
                                <div>
                                    <h3><?php echo esc_html( $company_data['name'] ?? '' ); ?></h3>
                                    <p><strong><?php esc_html_e( 'Code:', 'jadcoins-woo' ); ?></strong> <?php echo esc_html( get_option( 'jadcoins_company_code' ) ); ?></p>
                                    <p><strong><?php esc_html_e( 'API:', 'jadcoins-woo' ); ?></strong> <?php echo esc_html( get_option( 'jadcoins_api_url' ) ); ?></p>
                                    <?php if ( ! empty( $company_data['loyalty_settings'] ) ) : ?>
                                        <?php $ls = $company_data['loyalty_settings']; ?>
                                        <p><strong><?php esc_html_e( 'Points:', 'jadcoins-woo' ); ?></strong>
                                            <?php
                                            if ( ( $ls['points_type'] ?? '' ) === 'percent' ) {
                                                echo esc_html( ( $ls['percentage_points'] ?? 0 ) . '%' );
                                            } else {
                                                echo esc_html( $ls['fixed_points'] ?? 0 );
                                            }
                                            ?>
                                        </p>
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div class="jadcoins-disconnect-area">
                                <button type="button" id="jadcoins-disconnect" class="button jadcoins-btn-disconnect">
                                    🔌 <?php esc_html_e( 'Disconnect', 'jadcoins-woo' ); ?>
                                </button>
                                <span class="jadcoins-disconnect-hint"><?php esc_html_e( 'Disconnect to change API settings', 'jadcoins-woo' ); ?></span>
                            </div>
                        </div>

                        <!-- Hidden fields to preserve values -->
                        <input type="hidden" name="jadcoins_api_url" value="<?php echo esc_attr( get_option( 'jadcoins_api_url' ) ); ?>" />
                        <input type="hidden" name="jadcoins_company_code" value="<?php echo esc_attr( get_option( 'jadcoins_company_code' ) ); ?>" />
                        <input type="hidden" name="jadcoins_default_phone_code" value="<?php echo esc_attr( get_option( 'jadcoins_default_phone_code', '+972' ) ); ?>" />

                    <?php else : ?>
                        <!-- Disconnected State - editable -->
                        <table class="form-table">
                            <tr>
                                <th><label for="jadcoins_api_url"><?php esc_html_e( 'API URL', 'jadcoins-woo' ); ?></label></th>
                                <td>
                                    <input type="url" id="jadcoins_api_url" name="jadcoins_api_url"
                                           value="<?php echo esc_attr( get_option( 'jadcoins_api_url', 'https://jadcoins.online/api' ) ); ?>"
                                           class="regular-text" dir="ltr" />
                                </td>
                            </tr>
                            <tr>
                                <th><label for="jadcoins_company_code"><?php esc_html_e( 'Company Unique Code', 'jadcoins-woo' ); ?></label></th>
                                <td>
                                    <input type="text" id="jadcoins_company_code" name="jadcoins_company_code"
                                           value="<?php echo esc_attr( get_option( 'jadcoins_company_code', '' ) ); ?>"
                                           class="regular-text" dir="ltr" />
                                    <p class="description"><?php esc_html_e( 'Your company unique code from JadCoins dashboard.', 'jadcoins-woo' ); ?></p>
                                </td>
                            </tr>
                            <tr>
                                <th><label for="jadcoins_default_phone_code"><?php esc_html_e( 'Default Phone Code', 'jadcoins-woo' ); ?></label></th>
                                <td>
                                    <input type="text" id="jadcoins_default_phone_code" name="jadcoins_default_phone_code"
                                           value="<?php echo esc_attr( get_option( 'jadcoins_default_phone_code', '+972' ) ); ?>"
                                           class="small-text" dir="ltr" />
                                </td>
                            </tr>
                            <tr>
                                <th></th>
                                <td>
                                    <button type="button" id="jadcoins-test-connection" class="button button-primary jadcoins-btn-connect">
                                        🔗 <?php esc_html_e( 'Connect & Verify', 'jadcoins-woo' ); ?>
                                    </button>
                                    <span id="jadcoins-test-result" style="margin-left:10px;"></span>
                                </td>
                            </tr>
                        </table>

                        <div id="jadcoins-company-info" style="display:none;">
                            <div class="jadcoins-info-box">
                                <h3 id="jadcoins-company-name"></h3>
                                <p id="jadcoins-loyalty-status"></p>
                                <div id="jadcoins-loyalty-details"></div>
                            </div>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Loyalty Settings -->
                <div class="jadcoins-card">
                    <h2><?php esc_html_e( 'Loyalty Settings', 'jadcoins-woo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Enable Points Earning', 'jadcoins-woo' ); ?></th>
                            <td><label><input type="checkbox" name="jadcoins_enable_earn" value="1" <?php checked( get_option( 'jadcoins_enable_earn', 1 ) ); ?> />
                                <?php esc_html_e( 'Award points to customers when orders are completed', 'jadcoins-woo' ); ?></label></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Enable Wallet Payment', 'jadcoins-woo' ); ?></th>
                            <td><label><input type="checkbox" name="jadcoins_enable_wallet" value="1" <?php checked( get_option( 'jadcoins_enable_wallet', 1 ) ); ?> />
                                <?php esc_html_e( 'Allow customers to use wallet balance at checkout', 'jadcoins-woo' ); ?></label></td>
                        </tr>
                        <tr>
                            <th><label for="jadcoins_earn_order_status"><?php esc_html_e( 'Earn Points On Status', 'jadcoins-woo' ); ?></label></th>
                            <td>
                                <select id="jadcoins_earn_order_status" name="jadcoins_earn_order_status">
                                    <option value="completed" <?php selected( get_option( 'jadcoins_earn_order_status', 'completed' ), 'completed' ); ?>><?php esc_html_e( 'Completed', 'jadcoins-woo' ); ?></option>
                                    <option value="processing" <?php selected( get_option( 'jadcoins_earn_order_status', 'completed' ), 'processing' ); ?>><?php esc_html_e( 'Processing', 'jadcoins-woo' ); ?></option>
                                </select>
                            </td>
                        </tr>
                    </table>
                </div>

                <!-- Display Settings -->
                <div class="jadcoins-card">
                    <h2><?php esc_html_e( 'Display Settings', 'jadcoins-woo' ); ?></h2>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e( 'Show in My Account', 'jadcoins-woo' ); ?></th>
                            <td><label><input type="checkbox" name="jadcoins_show_in_myaccount" value="1" <?php checked( get_option( 'jadcoins_show_in_myaccount', 1 ) ); ?> />
                                <?php esc_html_e( 'Show JadCoins balance tab in My Account page', 'jadcoins-woo' ); ?></label></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Show Points on Product Page', 'jadcoins-woo' ); ?></th>
                            <td><label><input type="checkbox" name="jadcoins_show_on_product" value="1" <?php checked( get_option( 'jadcoins_show_on_product', 1 ) ); ?> />
                                <?php esc_html_e( 'Display how many points the customer will earn on product pages', 'jadcoins-woo' ); ?></label></td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e( 'Show Points in Cart', 'jadcoins-woo' ); ?></th>
                            <td><label><input type="checkbox" name="jadcoins_show_in_cart" value="1" <?php checked( get_option( 'jadcoins_show_in_cart', 1 ) ); ?> />
                                <?php esc_html_e( 'Display total points the customer will earn from their cart', 'jadcoins-woo' ); ?></label></td>
                        </tr>
                    </table>
                </div>

                <?php submit_button( __( 'Save Settings', 'jadcoins-woo' ) ); ?>
            </form>
        </div>
        <?php
    }
}
