<?php
/**
 * JadCoins My Account Integration - V2
 * Added: Birthday field, improved sync
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class JadCoins_Account {

    private static $instance = null;

    public static function instance() {
        if ( is_null( self::$instance ) ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        if ( ! get_option( 'jadcoins_show_in_myaccount', 1 ) ) {
            return;
        }

        add_action( 'init', array( $this, 'add_endpoint' ) );
        add_filter( 'woocommerce_account_menu_items', array( $this, 'add_menu_item' ) );
        add_action( 'woocommerce_account_jadcoins_endpoint', array( $this, 'endpoint_content' ) );
        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );

        // Sync on registration and profile save
        add_action( 'woocommerce_created_customer', array( $this, 'sync_customer_on_register' ), 10, 3 );
        add_action( 'woocommerce_save_account_details', array( $this, 'sync_customer_on_save' ), 10, 1 );

        // Birthday field in My Account edit
        add_action( 'woocommerce_edit_account_form', array( $this, 'add_birthday_field' ) );
        add_action( 'woocommerce_save_account_details', array( $this, 'save_birthday_field' ), 10, 1 );
        add_action( 'woocommerce_save_account_details_errors', array( $this, 'validate_birthday_field' ), 10, 1 );

        // Birthday field in checkout billing
        add_filter( 'woocommerce_billing_fields', array( $this, 'add_billing_fields' ) );
        add_action( 'woocommerce_checkout_update_order_meta', array( $this, 'save_checkout_birthday' ) );

        // Phone code field in billing
        add_filter( 'woocommerce_billing_fields', array( $this, 'add_phone_code_field' ) );
    }

    public function add_endpoint() {
        add_rewrite_endpoint( 'jadcoins', EP_ROOT | EP_PAGES );
    }

    public function add_menu_item( $items ) {
        $new_items = array();
        foreach ( $items as $key => $value ) {
            $new_items[ $key ] = $value;
            if ( $key === 'orders' ) {
                $new_items['jadcoins'] = __( 'JadCoins', 'jadcoins-woo' );
            }
        }
        return $new_items;
    }

    public function enqueue_assets() {
        if ( is_account_page() ) {
            wp_enqueue_style( 'jadcoins-account', JADCOINS_WOO_PLUGIN_URL . 'assets/css/account.css', array(), JADCOINS_WOO_VERSION );
        }
    }

    /**
     * My Account endpoint content
     */
    public function endpoint_content() {
        $api     = JadCoins_API::instance();
        $user_id = get_current_user_id();

        if ( ! $api->is_configured() ) {
            echo '<div class="woocommerce-message woocommerce-message--info">';
            esc_html_e( 'Loyalty program is not currently available.', 'jadcoins-woo' );
            echo '</div>';
            return;
        }

        $phone_data = JadCoins_API::get_user_phone( $user_id );
        if ( empty( $phone_data['phone'] ) ) {
            echo '<div class="woocommerce-message woocommerce-message--info">';
            esc_html_e( 'Please add your phone number in your billing details to view your JadCoins balance.', 'jadcoins-woo' );
            echo '</div>';
            return;
        }

        $user     = get_userdata( $user_id );
        $birthday = JadCoins_API::get_user_birthday( $user_id );
        $result   = $api->check_or_create_customer(
            $phone_data['phone'],
            $phone_data['phone_code'],
            $user->display_name,
            $user->user_email,
            $birthday
        );

        if ( empty( $result['value'] ) || empty( $result['data'] ) ) {
            echo '<div class="woocommerce-message woocommerce-message--error">';
            esc_html_e( 'Could not retrieve your JadCoins information. Please try again later.', 'jadcoins-woo' );
            echo '</div>';
            return;
        }

        $data     = $result['data'];
        $wallets  = $data['wallets'] ?? array();
        $currency = $data['currency'] ?? '₪';
        $loyalty  = JadCoins_API::get_loyalty_settings();
        ?>
        <div class="jadcoins-account-section">
            <div class="jadcoins-account-header">
                <h2><span class="jadcoins-star"><?php echo JadCoins_API::get_icon_html( 28 ); ?></span> <?php esc_html_e( 'My JadCoins', 'jadcoins-woo' ); ?></h2>
            </div>

            <div class="jadcoins-cards-grid">
                <div class="jadcoins-card jadcoins-card-points">
                    <div class="jadcoins-card-icon">🪙</div>
                    <div class="jadcoins-card-label"><?php esc_html_e( 'Points Balance', 'jadcoins-woo' ); ?></div>
                    <div class="jadcoins-card-value"><?php echo esc_html( $wallets['points_balance'] ?? 0 ); ?></div>
                    <?php if ( ! empty( $wallets['points_value'] ) ) : ?>
                        <div class="jadcoins-card-sub">
                            <?php printf( esc_html__( 'Worth %1$s%2$s', 'jadcoins-woo' ), esc_html( $currency ), esc_html( $wallets['points_value'] ) ); ?>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="jadcoins-card jadcoins-card-wallet">
                    <div class="jadcoins-card-icon">💰</div>
                    <div class="jadcoins-card-label"><?php esc_html_e( 'Wallet Balance', 'jadcoins-woo' ); ?></div>
                    <div class="jadcoins-card-value"><?php echo esc_html( $currency . number_format( (float) ( $wallets['wallet_balance'] ?? 0 ), 2 ) ); ?></div>
                    <div class="jadcoins-card-sub"><?php esc_html_e( 'Available at checkout', 'jadcoins-woo' ); ?></div>
                </div>
            </div>

            <?php if ( ! empty( $loyalty ) ) : ?>
            <div class="jadcoins-how-it-works">
                <h3><?php esc_html_e( 'How It Works', 'jadcoins-woo' ); ?></h3>
                <div class="jadcoins-steps">
                    <div class="jadcoins-step">
                        <span class="jadcoins-step-num">1</span>
                        <span class="jadcoins-step-text">
                            <?php
                            if ( ( $loyalty['points_type'] ?? '' ) === 'percent' ) {
                                printf( esc_html__( 'Earn %s%% points on every purchase', 'jadcoins-woo' ), esc_html( $loyalty['percentage_points'] ?? 0 ) );
                            } else {
                                printf( esc_html__( 'Earn %s points on every purchase', 'jadcoins-woo' ), esc_html( $loyalty['fixed_points'] ?? 0 ) );
                            }
                            ?>
                        </span>
                    </div>
                    <div class="jadcoins-step">
                        <span class="jadcoins-step-num">2</span>
                        <span class="jadcoins-step-text">
                            <?php
                            $min_points  = floatval( $loyalty['min_redemption_points'] ?? 10 );
                            $point_value = floatval( $loyalty['point_value'] ?? 1 );
                            printf(
                                esc_html__( 'Every %1$s points = %2$s in your wallet', 'jadcoins-woo' ),
                                esc_html( $min_points ),
                                wp_kses_post( wc_price( $min_points * $point_value ) )
                            );
                            ?>
                        </span>
                    </div>
                    <div class="jadcoins-step">
                        <span class="jadcoins-step-num">3</span>
                        <span class="jadcoins-step-text"><?php esc_html_e( 'Use your wallet balance at checkout for discounts!', 'jadcoins-woo' ); ?></span>
                    </div>
                </div>
            </div>
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Birthday field in My Account edit form
     */
    public function add_birthday_field() {
        $user_id  = get_current_user_id();
        $birthday = get_user_meta( $user_id, 'jadcoins_birthday', true );
        ?>
        <p class="woocommerce-form-row woocommerce-form-row--wide form-row form-row-wide">
            <label for="jadcoins_birthday"><?php esc_html_e( 'Birthday', 'jadcoins-woo' ); ?>&nbsp;<span class="required">*</span></label>
            <input type="date" class="woocommerce-Input input-text" name="jadcoins_birthday" id="jadcoins_birthday"
                   value="<?php echo esc_attr( $birthday ); ?>" required />
        </p>
        <?php
    }

    /**
     * Save birthday on account save
     */
    public function save_birthday_field( $user_id ) {
        if ( isset( $_POST['jadcoins_birthday'] ) ) {
            $birthday = sanitize_text_field( $_POST['jadcoins_birthday'] );
            update_user_meta( $user_id, 'jadcoins_birthday', $birthday );
        }
    }

    /**
     * Validate birthday field on account save
     */
    public function validate_birthday_field( $errors ) {
        if ( empty( $_POST['jadcoins_birthday'] ) ) {
            $errors->add( 'jadcoins_birthday_required', __( 'Please enter your birthday.', 'jadcoins-woo' ) );
        }
    }

    /**
     * Add birthday field to checkout billing
     */
    public function add_billing_fields( $fields ) {
        // Ensure phone field is required
        if ( isset( $fields['billing_phone'] ) ) {
            $fields['billing_phone']['required'] = true;
        } else {
            $fields['billing_phone'] = array(
                'type'     => 'tel',
                'label'    => __( 'Phone', 'jadcoins-woo' ),
                'required' => true,
                'class'    => array( 'form-row-wide' ),
                'priority' => 100,
            );
        }

        $fields['billing_jadcoins_birthday'] = array(
            'type'     => 'date',
            'label'    => __( 'Birthday', 'jadcoins-woo' ),
            'required' => true,
            'class'    => array( 'form-row-first' ),
            'priority' => 25,
        );
        return $fields;
    }

    /**
     * Save birthday from checkout
     */
    public function save_checkout_birthday( $order_id ) {
        if ( ! empty( $_POST['billing_jadcoins_birthday'] ) ) {
            $birthday = sanitize_text_field( $_POST['billing_jadcoins_birthday'] );
            $order    = wc_get_order( $order_id );
            if ( $order && $order->get_customer_id() ) {
                update_user_meta( $order->get_customer_id(), 'jadcoins_birthday', $birthday );
            }
        }
    }

    /**
     * Sync customer
     */
    public function sync_customer_on_register( $customer_id, $new_customer_data, $password_generated ) {
        $this->sync_customer( $customer_id );
    }

    public function sync_customer_on_save( $user_id ) {
        $this->sync_customer( $user_id );
    }

    private function sync_customer( $user_id ) {
        $api = JadCoins_API::instance();
        if ( ! $api->is_configured() ) {
            return;
        }
        $phone_data = JadCoins_API::get_user_phone( $user_id );
        if ( empty( $phone_data['phone'] ) ) {
            return;
        }
        $user     = get_userdata( $user_id );
        $birthday = JadCoins_API::get_user_birthday( $user_id );
        $api->check_or_create_customer(
            $phone_data['phone'],
            $phone_data['phone_code'],
            $user->display_name,
            $user->user_email,
            $birthday
        );
    }

    /**
     * Add phone code field to billing
     */
    public function add_phone_code_field( $fields ) {
        $fields['billing_jadcoins_phone_code'] = array(
            'type'        => 'hidden',
            'required'    => false,
            'class'       => array( 'hidden' ),
            'priority'    => 24,
            'default'     => get_option( 'jadcoins_default_phone_code', '+972' ),
        );
        return $fields;
    }
}
