(function($) {
    'use strict';

    var JadCoinsCheckout = {
        walletBalance: 0,
        pointsBalance: 0,
        currency: '',
        isLoading: false,

        init: function() {
            this.bindEvents();
            this.loadBalance();
        },

        bindEvents: function() {
            $(document).on('click', '#jadcoins-apply-wallet', this.applyWallet.bind(this));
            $(document).on('click', '#jadcoins-remove-wallet', this.removeWallet.bind(this));

            // Reload balance when checkout updates
            $(document.body).on('updated_checkout', function() {
                // Don't reload if we already have data
            });
        },

        loadBalance: function() {
            var self = this;

            if (self.isLoading) return;
            self.isLoading = true;

            $('#jadcoins-wallet-loading').show();
            $('#jadcoins-wallet-balance-info, #jadcoins-wallet-form, #jadcoins-wallet-unavailable').hide();

            $.ajax({
                url: jadcoins_checkout.ajax_url,
                type: 'POST',
                data: {
                    action: 'jadcoins_get_balance',
                    nonce: jadcoins_checkout.nonce,
                },
                success: function(response) {
                    $('#jadcoins-wallet-loading').hide();

                    if (response.success) {
                        var data = response.data;
                        self.walletBalance = parseFloat(data.wallet_balance) || 0;
                        self.pointsBalance = parseInt(data.points_balance) || 0;
                        self.currency = data.currency || '₪';

                        $('#jadcoins-points-balance').text(self.pointsBalance);
                        $('#jadcoins-wallet-balance').text(self.currency + self.walletBalance.toFixed(2));

                        $('#jadcoins-wallet-balance-info').show();

                        if (self.walletBalance > 0) {
                            $('#jadcoins-wallet-form').show();
                            $('#jadcoins-wallet-amount').attr('max', self.walletBalance);
                        } else {
                            $('#jadcoins-wallet-unavailable').show();
                        }
                    } else {
                        $('#jadcoins-wallet-unavailable').show();
                    }
                },
                error: function() {
                    $('#jadcoins-wallet-loading').hide();
                    $('#jadcoins-wallet-unavailable').show();
                },
                complete: function() {
                    self.isLoading = false;
                }
            });
        },

        applyWallet: function(e) {
            e.preventDefault();

            var self = this;
            var amount = parseFloat($('#jadcoins-wallet-amount').val());

            if (!amount || amount <= 0) {
                self.showMessage(jadcoins_checkout.i18n.enter_amount, 'error');
                return;
            }

            if (amount > self.walletBalance) {
                self.showMessage(jadcoins_checkout.i18n.max_exceeded, 'error');
                return;
            }

            var $btn = $('#jadcoins-apply-wallet');
            $btn.prop('disabled', true).text(jadcoins_checkout.i18n.loading);

            $.ajax({
                url: jadcoins_checkout.ajax_url,
                type: 'POST',
                data: {
                    action: 'jadcoins_apply_wallet',
                    nonce: jadcoins_checkout.nonce,
                    amount: amount,
                },
                success: function(response) {
                    if (response.success) {
                        self.showMessage(jadcoins_checkout.i18n.applied, 'success');

                        $('#jadcoins-applied-amount').html(response.data.formatted);
                        $('#jadcoins-wallet-form').hide();
                        $('#jadcoins-wallet-applied').show();

                        // Trigger WooCommerce checkout refresh
                        $(document.body).trigger('update_checkout');
                    } else {
                        self.showMessage(response.data, 'error');
                    }
                },
                error: function() {
                    self.showMessage(jadcoins_checkout.i18n.error, 'error');
                },
                complete: function() {
                    $btn.prop('disabled', false).text(jadcoins_checkout.i18n.apply);
                }
            });
        },

        removeWallet: function(e) {
            e.preventDefault();

            var self = this;
            var $btn = $('#jadcoins-remove-wallet');
            $btn.prop('disabled', true);

            $.ajax({
                url: jadcoins_checkout.ajax_url,
                type: 'POST',
                data: {
                    action: 'jadcoins_remove_wallet',
                    nonce: jadcoins_checkout.nonce,
                },
                success: function(response) {
                    if (response.success) {
                        self.showMessage(jadcoins_checkout.i18n.removed, 'success');

                        $('#jadcoins-wallet-applied').hide();
                        $('#jadcoins-wallet-amount').val('');
                        if (self.walletBalance > 0) {
                            $('#jadcoins-wallet-form').show();
                        }

                        $(document.body).trigger('update_checkout');
                    }
                },
                complete: function() {
                    $btn.prop('disabled', false);
                }
            });
        },

        showMessage: function(msg, type) {
            var $el = $('#jadcoins-wallet-message');
            $el.text(msg).removeClass('success error').addClass(type).show();

            setTimeout(function() {
                $el.fadeOut();
            }, 4000);
        }
    };

    $(document).ready(function() {
        if ($('#jadcoins-wallet-section').length) {
            JadCoinsCheckout.init();
        }
    });

})(jQuery);
